import struct
from io import StringIO
from . import asciiparser
from mathutils import Matrix


class MeshLOD:
    __slots__ = [
        "maxDist",
        "flags",
        "vertices",
        "normals",
        "uvmaps",
        "surfaces",
        "weightMaps",
        "morphMaps",
        "vertexWeights",
    ]

    ML_HALF_FACE_FORWARD = 1
    ML_FULL_FACE_FORWARD = 2

    def __init__(self):
        self.maxDist = 0.0
        self.flags = 0
        self.vertices = []      # [(x, y, z), ...]
        self.normals = []       # [(nx, ny, nz), ...]
        self.uvmaps = []        # [MeshUVMap, ...]
        self.surfaces = []      # [MeshSurface, ...]
        self.weightMaps = []    # [MeshWeightMap, ...]
        self.morphMaps = []     # [MeshMorphMap, ...]
        self.vertexWeights = []  # [((i1, i2, i3, i4), (w1, w2, w3, w4)), ...]


class MeshUVMap:
    __slots__ = [
        "name",
        "coords",
    ]

    def __init__(self):
        self.name = ""
        self.coords = []    # [(u, v), ...]


class MeshSurface:
    __slots__ = [
        "firstVtx",
        "vtxCount",
        "name",
        "tris",
        "weightMaps",
    ]

    def __init__(self):
        self.firstVtx = 0
        self.vtxCount = 0
        self.name = ""
        self.tris = []  # [(vtx1, vtx2, vtx3), ...]
        self.weightMaps = []  # [wmapIdx, ...]


class MeshWeightMap:
    __slots__ = [
        "name",
        "values",
    ]

    def __init__(self):
        self.name = ""
        self.values = dict()    # {vtx: weight, ...}


class MeshMorphMap:
    __slots__ = [
        "name",
        "relative",
        "values",
    ]

    def __init__(self):
        self.name = ""
        self.relative = False
        self.values = dict()    # {vtx: (x, y, z, nx, ny, nz), ...}


class SkeletonLOD:
    __slots__ = [
        "maxDist",
        "bones",
    ]

    def __init__(self):
        self.maxDist = 0.0
        self.bones = []  # [SkeletonBone, ...]

    def sortBones(self):
        lst = []

        def _sortBones(parentName: str = ""):
            for sb in self.bones:
                if (sb.parentName == parentName):
                    lst.append(sb)
                    if (sb.name):
                        _sortBones(sb.name)

        _sortBones()
        self.bones = lst


class SkeletonBone:
    __slots__ = [
        "name",
        "parentName",
        "absPlacement",
        "relPlacement",
        "offsetLen",
        "length"
    ]

    def __init__(self):
        self.name = ""
        self.parentName = ""
        self.absPlacement = [[], [], []]    # mat 3x4
        self.relPlacement = [[], []]    # [(x, y, z), (w, x, y, z)]
        self.offsetLen = 0.0
        self.length = 0.0


class Animation:
    __slots__ = [
        "name",
        "frameCount",
        "secPerFrame",
        "treshold",
        "morphEvps",
        "boneEvps",
    ]

    def __init__(self):
        self.name = ""
        self.frameCount = 0
        self.secPerFrame = 0.0
        self.treshold = 0.0
        self.morphEvps = []  # [MorphEnvelope, ...]
        self.boneEvps = []  # [BoneEnvelope, ...]


class MorphEnvelope:
    __slots__ = [
        "mapName",
        "factors"
    ]

    def __init__(self):
        self.mapName = ""
        self.factors = []   # [f1, ...]


class BoneEnvelope:
    __slots__ = [
        "boneName",
        "defPos",
        "positions",
        "rotations",
        "offsetLen",
    ]

    def __init__(self):
        self.boneName = ""
        self.defPos = [[], [], []]  # mat 3x4
        self.positions = dict()  # {frameNo: (x, y, z), ...}
        self.rotations = dict()  # {frameNo: (w, x, y, z), ...}
        self.offsetLen = 0.0


def readString(file) -> str:
    strLen = struct.unpack("I", file.read(4))[0]
    if (strLen):
        return file.read(strLen).decode("cp1251")
    return ""


def writeString(file, string: str):
    file.write(struct.pack("I", len(string)))
    if (string):
        file.write(string.encode("cp1251"))


def readBinFileName(file) -> str:
    magic = file.read(4)
    if (magic != b"DFNM"):
        raise ValueError("Not a binary file name")

    return readString(file)


def readMeshBin(filename: str) -> list:
    lods = []

    with open(filename, "rb") as file:
        magic = file.read(4)
        if (magic != b"MESH"):
            raise ValueError("Invalid binary mesh file")

        ver = struct.unpack("i", file.read(4))[0]
        lcVer = ver >> 16
        if (not (ver in {11, 12, 16, 17, 18, 19} or lcVer in {18, 19})):
            raise ValueError("Invalid binary mesh file version {}".format(ver))

        if (lcVer >= 18):
            return readMeshBinLC(file, lcVer)
        if (ver > 16):
            return readMeshBinLC(file, ver)

        vtxFormat = "3f4x"
        vtxSize = 16
        triFormat = "3i"
        triSize = 12
        morphSetFormat = "6f4x"
        morphSetSize = 28

        if (ver > 12):
            vtxFormat = "3f"
            vtxSize = 12
            triFormat = "3H"
            triSize = 6
            morphSetFormat = "6f"
            morphSetSize = 24

            # file size
            file.read(4)

        lodCount = struct.unpack("i", file.read(4))[0]

        for i in range(lodCount):
            lod = MeshLOD()
            lods.append(lod)

            if (ver == 16):
                vtxCount = struct.unpack("i", file.read(4))[0]
                uvmapCount = struct.unpack("i", file.read(4))[0]
                surfCount = struct.unpack("i", file.read(4))[0]
                wmapCount = struct.unpack("i", file.read(4))[0]
                mmapCount = struct.unpack("i", file.read(4))[0]
                vertexWeightCount = struct.unpack("i", file.read(4))[0]

            readString(file)

            lod.maxDist = struct.unpack("f", file.read(4))[0]
            lod.flags = struct.unpack("I", file.read(4))[0]

            if (lod.flags == 0xCDCDCDCD):
                lod.flags = 0

            if (ver < 16):
                vtxCount = struct.unpack("i", file.read(4))[0]

            lod.vertices = [struct.unpack(vtxFormat, file.read(vtxSize)) for j in range(vtxCount)]
            lod.normals = [struct.unpack(vtxFormat, file.read(vtxSize)) for j in range(vtxCount)]

            if (ver < 16):
                uvmapCount = struct.unpack("i", file.read(4))[0]

            for j in range(uvmapCount):
                uvMap = MeshUVMap()
                lod.uvmaps.append(uvMap)

                uvMap.name = readString(file)

                uvMap.coords = [struct.unpack("2f", file.read(8)) for k in range(vtxCount)]

            if (ver < 16):
                surfCount = struct.unpack("i", file.read(4))[0]

            for j in range(surfCount):
                surf = MeshSurface()
                lod.surfaces.append(surf)

                surf.name = readString(file)

                if (ver == 16):
                    # unk
                    struct.unpack("i", file.read(4))[0]

                surf.firstVtx = struct.unpack("i", file.read(4))[0]
                surf.vtxCount = struct.unpack("i", file.read(4))[0]

                triCount = struct.unpack("i", file.read(4))[0]
                surf.tris = [struct.unpack(triFormat, file.read(triSize)) for k in range(triCount)]

                if (ver == 16):
                    surfWmapCount = struct.unpack("i", file.read(4))[0]
                    surf.weightMaps = [struct.unpack("B", file.read(1))[0] for k in range(surfWmapCount)]

                shaderExists = struct.unpack("i", file.read(4))[0]
                if (shaderExists):
                    cttx = struct.unpack("i", file.read(4))[0]
                    cttc = struct.unpack("i", file.read(4))[0]
                    ctcol = struct.unpack("i", file.read(4))[0]
                    ctfl = struct.unpack("i", file.read(4))[0]

                    readString(file)

                    for k in range(cttx):
                        readString(file)

                    file.seek(4*cttc + 4*ctcol + 4*ctfl, 1)
                    if (ver > 11):
                        file.seek(4, 1)

            if (ver < 16):
                wmapCount = struct.unpack("i", file.read(4))[0]

            vtxWeights = [0.0 for i in range(vtxCount)]

            for j in range(wmapCount):
                wmap = MeshWeightMap()
                lod.weightMaps.append(wmap)

                wmap.name = readString(file)

                weightCount = struct.unpack("i", file.read(4))[0]
                for k in range(weightCount):
                    vtx = struct.unpack("i", file.read(4))[0]
                    value = struct.unpack("f", file.read(4))[0]
                    wmap.values[vtx] = value

                    vtxWeights[vtx] += value

            if (ver < 16):
                mmapCount = struct.unpack("i", file.read(4))[0]

            for j in range(mmapCount):
                mmap = MeshMorphMap()
                lod.morphMaps.append(mmap)

                mmap.name = readString(file)

                mmap.relative = bool(struct.unpack("i", file.read(4))[0])

                setCount = struct.unpack("i", file.read(4))[0]
                for k in range(setCount):
                    vtx = struct.unpack("i", file.read(4))[0]
                    mmap.values[vtx] = struct.unpack(morphSetFormat, file.read(morphSetSize))

            if (ver == 16):
                lod.vertexWeights = [(struct.unpack("4B", file.read(4)), struct.unpack("4B", file.read(4))) for j in range(vertexWeightCount)]

    return lods

def decodeLCint(n: int, key: int) -> tuple:
    key = (key + 13) & 0xFF
    m = key << 24

    key = (key + 23) & 0xFF
    m |= (key << 16)

    key = (key + 19) & 0xFF
    m |= (key << 8)

    key = (key + 29) & 0xFF
    m |= key

    key = (key + 5) & 0xFF
    res = (m ^ n) & 0xFFFFFFFF
    res = (res ^ 0x80000000) - 0x80000000

    return (res, key)

def readMeshBinLC(file, ver: int) -> list:
    lods = []

    vtxFormat = "3f"
    vtxSize = 12
    triFormat = "3H"
    triSize = 6
    morphSetFormat = "6f"
    morphSetSize = 24

    if (ver == 18):
        file.read(8)
    else:
        # file size
        file.read(4)
        if (ver == 19):
            file.read(12)

    keyLC = 17

    lodCount = struct.unpack("i", file.read(4))[0] 
    lodCount, keyLC = decodeLCint(lodCount, keyLC)

    for i in range(lodCount):
        lod = MeshLOD()
        lods.append(lod)

        vtxCount = struct.unpack("I", file.read(4))[0]
        wmapCount = struct.unpack("I", file.read(4))[0]
        uvmapCount = struct.unpack("I", file.read(4))[0]
        vertexWeightCount = struct.unpack("I", file.read(4))[0]
        surfCount = struct.unpack("I", file.read(4))[0]
        mmapCount = struct.unpack("I", file.read(4))[0]

        vtxCount, keyLC = decodeLCint(vtxCount, keyLC)
        wmapCount, keyLC = decodeLCint(wmapCount, keyLC)
        uvmapCount, keyLC = decodeLCint(uvmapCount, keyLC)
        vertexWeightCount, keyLC = decodeLCint(vertexWeightCount, keyLC)
        surfCount, keyLC = decodeLCint(surfCount, keyLC)
        mmapCount, keyLC = decodeLCint(mmapCount, keyLC)

        readString(file)

        lod.maxDist = struct.unpack("f", file.read(4))[0]
        lod.flags = struct.unpack("i", file.read(4))[0]
        lod.flags, keyLC = decodeLCint(lod.flags, keyLC)

        if (lod.flags == 0xCDCDCDCD):
            lod.flags = 0

        lod.vertices = [struct.unpack(vtxFormat, file.read(vtxSize)) for j in range(vtxCount)]
        lod.normals = [struct.unpack(vtxFormat, file.read(vtxSize)) for j in range(vtxCount)]

        for j in range(uvmapCount):
            uvMap = MeshUVMap()
            lod.uvmaps.append(uvMap)

            uvMap.name = readString(file)

            uvMap.coords = [struct.unpack("2f", file.read(8)) for k in range(vtxCount)]

        for j in range(surfCount):
            surf = MeshSurface()
            lod.surfaces.append(surf)

            surf.firstVtx = struct.unpack("i", file.read(4))[0]
            surf.vtxCount = struct.unpack("i", file.read(4))[0]

            surf.firstVtx, keyLC = decodeLCint(surf.firstVtx, keyLC)
            surf.vtxCount, keyLC = decodeLCint(surf.vtxCount, keyLC)

            triCount = struct.unpack("i", file.read(4))[0]
            triCount, keyLC = decodeLCint(triCount, keyLC)

            surf.tris = [struct.unpack(triFormat, file.read(triSize)) for k in range(triCount)]

            surf.name = readString(file)

            # unk
            unk = struct.unpack("i", file.read(4))[0]

            unk, keyLC = decodeLCint(unk, keyLC)

            surfWmapCount = struct.unpack("i", file.read(4))[0]
            surfWmapCount, keyLC = decodeLCint(surfWmapCount, keyLC)

            surf.weightMaps = [struct.unpack("B", file.read(1))[0] for k in range(surfWmapCount)]

            shaderExists = struct.unpack("i", file.read(4))[0]
            shaderExists, keyLC = decodeLCint(shaderExists, keyLC)

            if (shaderExists):
                ctcol = struct.unpack("i", file.read(4))[0]
                ctfl = struct.unpack("i", file.read(4))[0]
                cttx = struct.unpack("i", file.read(4))[0]
                cttc = struct.unpack("i", file.read(4))[0]

                ctcol, keyLC = decodeLCint(ctcol, keyLC)
                ctfl, keyLC = decodeLCint(ctfl, keyLC)
                cttx, keyLC = decodeLCint(cttx, keyLC)
                cttc, keyLC = decodeLCint(cttc, keyLC)

                readString(file)

                for k in range(cttx):
                    readString(file)

                v = struct.unpack("i", file.read(4))[0]
                v, keyLC = decodeLCint(v, keyLC)

                for k in range(ctcol):
                    v = struct.unpack("i", file.read(4))[0]
                    v, keyLC = decodeLCint(v, keyLC)

                file.read(4*ctfl)

                for k in range(cttc):
                    v = struct.unpack("i", file.read(4))[0]
                    v, keyLC = decodeLCint(v, keyLC)

        vtxWeights = [0.0 for i in range(vtxCount)]

        for j in range(wmapCount):
            wmap = MeshWeightMap()
            lod.weightMaps.append(wmap)

            wmap.name = readString(file)

            weightCount = struct.unpack("i", file.read(4))[0]
            weightCount, keyLC = decodeLCint(weightCount, keyLC)

            for k in range(weightCount):
                vtx = struct.unpack("i", file.read(4))[0]
                value = struct.unpack("f", file.read(4))[0]
                wmap.values[vtx] = value
                
                vtxWeights[vtx] += value

        for j in range(mmapCount):
            mmap = MeshMorphMap()
            lod.morphMaps.append(mmap)

            mmap.name = readString(file)

            rel = struct.unpack("i", file.read(4))[0]
            rel, keyLC = decodeLCint(rel, keyLC)

            mmap.relative = bool(rel)

            setCount = struct.unpack("i", file.read(4))[0]
            setCount, keyLC = decodeLCint(setCount, keyLC)
            for k in range(setCount):
                vtx = struct.unpack("i", file.read(4))[0]
                mmap.values[vtx] = struct.unpack(morphSetFormat, file.read(morphSetSize))

        lod.vertexWeights = [(struct.unpack("4B", file.read(4)), struct.unpack("4B", file.read(4))) for j in range(vertexWeightCount)]

    return lods

def writeMeshBin(meshlods: list, filename: str, ver: int):
    with open(filename, "wb") as file:
        file.write(b"MESH")
        file.write(struct.pack("i", ver))

        vtxFormat = "3f4x"
        triFormat = "3i"
        morphSetFormat = "6f4x"

        if (ver > 12):
            vtxFormat = "3f"
            triFormat = "3H"
            morphSetFormat = "6f"

            fileSizePos = file.tell()
            file.write(b"\x00\x00\x00\x00")

        file.write(struct.pack("i", len(meshlods)))
        for lod in meshlods:
            if (ver > 12):
                file.write(
                    struct.pack(
                        "6i",
                        len(lod.vertices),
                        len(lod.uvmaps),
                        len(lod.surfaces),
                        len(lod.weightMaps),
                        len(lod.morphMaps),
                        len(lod.vertexWeights)
                    )
                )

            writeString(file, "")   # no source filename
            file.write(struct.pack("f", lod.maxDist))
            file.write(struct.pack("I", lod.flags))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.vertices)))
            for vtx in lod.vertices:
                file.write(struct.pack(vtxFormat, *vtx))
            for norm in lod.normals:
                file.write(struct.pack(vtxFormat, *norm))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.uvmaps)))
            for uvmap in lod.uvmaps:
                file.write(struct.pack("i", len(uvmap.name)))
                file.write(uvmap.name.encode())

                for uv in uvmap.coords:
                    file.write(struct.pack("2f", *uv))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.surfaces)))
            for surf in lod.surfaces:
                writeString(file, surf.name)

                if (ver > 12):
                    # unk
                    file.write(b"\x00\x00\x00\x00")

                file.write(struct.pack("i", surf.firstVtx))
                file.write(struct.pack("i", surf.vtxCount))
                file.write(struct.pack("i", len(surf.tris)))
                for tri in surf.tris:
                    file.write(struct.pack(triFormat, *tri))

                if (ver > 12):
                    file.write(struct.pack("i", len(surf.weightMaps)))
                    for idx in surf.weightMaps:
                        file.write(struct.pack("B", idx))

                file.write(struct.pack("i", 0))  # no shader exists

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.weightMaps)))
            for wmap in lod.weightMaps:
                writeString(file, wmap.name)

                file.write(struct.pack("i", len(wmap.values)))
                for vtx, w in wmap.values.items():
                    file.write(struct.pack("if", vtx, w))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.morphMaps)))
            for mmap in lod.morphMaps:
                writeString(file, mmap.name)

                file.write(struct.pack("i", int(mmap.relative)))

                file.write(struct.pack("i", len(mmap.values)))
                for vtx, val in mmap.values.items():
                    file.write(struct.pack("i" + morphSetFormat, vtx, *val))

            if (ver > 12):
                for indices, weights in lod.vertexWeights:
                    file.write(struct.pack("8B", *indices, *weights))

        if (ver > 12):
            fileSize = file.tell()
            file.seek(fileSizePos)
            file.write(struct.pack("I", fileSize))

def writeMeshBinLC(meshlods: list, filename: str, ver: int):
    with open(filename, "wb") as file:
        file.write(b"MESH")
        file.write(struct.pack("i", ver))

        vtxFormat = "3f4x"
        triFormat = "3i"
        morphSetFormat = "6f4x"

        if (ver > 12):
            vtxFormat = "3f"
            triFormat = "3H"
            morphSetFormat = "6f"

            fileSizePos = file.tell()
            file.write(b"\x00\x00\x00\x00")

        keyLC = 17
        lodCount, keyLC = decodeLCint(len(meshlods), keyLC)

        file.write(struct.pack("i", lodCount))
        for lod in meshlods:
            if (ver > 12):
                vtxCount, keyLC = decodeLCint(len(lod.vertices), keyLC)
                wmapCount, keyLC = decodeLCint(len(lod.weightMaps), keyLC)
                uvCount, keyLC = decodeLCint(len(lod.uvmaps), keyLC)
                multiWeightCount, keyLC = decodeLCint(len(lod.vertexWeights), keyLC)
                surfCount, keyLC = decodeLCint(len(lod.surfaces), keyLC)
                morphCount, keyLC = decodeLCint(len(lod.morphMaps), keyLC)
                file.write(
                    struct.pack(
                        "6i",
                        vtxCount,
                        wmapCount,
                        uvCount,
                        multiWeightCount,
                        surfCount,
                        morphCount,
                    )
                )

            writeString(file, "")   # no source filename
            file.write(struct.pack("f", lod.maxDist))
            flags, keyLC = decodeLCint(lod.flags, keyLC)
            file.write(struct.pack("i", flags))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.vertices)))
            for vtx in lod.vertices:
                file.write(struct.pack(vtxFormat, *vtx))
            for norm in lod.normals:
                file.write(struct.pack(vtxFormat, *norm))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.uvmaps)))
            for uvmap in lod.uvmaps:
                file.write(struct.pack("i", len(uvmap.name)))
                file.write(uvmap.name.encode())

                for uv in uvmap.coords:
                    file.write(struct.pack("2f", *uv))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.surfaces)))
            for surf in lod.surfaces:
                surfFirstVtx, keyLC = decodeLCint(surf.firstVtx, keyLC)
                surfVtxCount, keyLC = decodeLCint(surf.vtxCount, keyLC)
                surfTriCount, keyLC = decodeLCint(len(surf.tris), keyLC)

                file.write(struct.pack("i", surfFirstVtx))
                file.write(struct.pack("i", surfVtxCount))
                file.write(struct.pack("i", surfTriCount))
                for tri in surf.tris:
                    file.write(struct.pack(triFormat, *tri))

                writeString(file, surf.name)

                if (ver > 12):
                    unk, keyLC = decodeLCint(0, keyLC)
                    surfWmapCount, keyLC = decodeLCint(len(surf.weightMaps), keyLC)

                    file.write(struct.pack("i", unk))
                    file.write(struct.pack("i", surfWmapCount))
                    for idx in surf.weightMaps:
                        file.write(struct.pack("B", idx))

                shaderExists, keyLC = decodeLCint(0, keyLC)
                file.write(struct.pack("i", shaderExists))  # no shader exists

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.weightMaps)))
            for wmap in lod.weightMaps:
                writeString(file, wmap.name)

                weightCount, keyLC = decodeLCint(len(wmap.values), keyLC)
                file.write(struct.pack("i", weightCount))
                for vtx, w in wmap.values.items():
                    file.write(struct.pack("if", vtx, w))

            if (ver <= 12):
                file.write(struct.pack("i", len(lod.morphMaps)))

            for mmap in lod.morphMaps:
                writeString(file, mmap.name)

                rel, keyLC = decodeLCint(int(mmap.relative), keyLC)
                file.write(struct.pack("i", rel))

                setCount, keyLC = decodeLCint(len(mmap.values), keyLC)
                file.write(struct.pack("i", setCount))
                for vtx, val in mmap.values.items():
                    file.write(struct.pack("i" + morphSetFormat, vtx, *val))

            if (ver > 12):
                for indices, weights in lod.vertexWeights:
                    file.write(struct.pack("8B", *indices, *weights))

        if (ver > 12):
            fileSize = file.tell()
            file.seek(fileSizePos)
            file.write(struct.pack("I", fileSize))

def readMeshAscii(filename: str) -> MeshLOD:
    lod = MeshLOD()

    with open(filename, "r") as file:
        parser = asciiparser.AsciiParser(StringIO(file.read()))

    ver = parser.getFloat("SE_MESH")
    if (ver != 0.1):
        raise ValueError("Invalid ascii mesh file version {}".format(ver))


    lod.vertices = [parser.scanf("%f,%f,%f;") for i in parser.iterList("VERTICES")]
    lod.normals = [parser.scanf("%f,%f,%f;") for i in parser.iterList("NORMALS")]

    for i in parser.iterList("UVMAPS"):
        parser.beginBlock()

        uvmap = MeshUVMap()
        lod.uvmaps.append(uvmap)
        uvmap.name = parser.getString("NAME")

        uvmap.coords = [parser.scanf("%f,%f;") for j in parser.iterList("TEXCOORDS")]

        parser.endBlock()

    for i in parser.iterList("SURFACES"):
        parser.beginBlock()

        surf = MeshSurface()
        lod.surfaces.append(surf)
        surf.name = parser.getString("NAME")

        surf.tris = [parser.scanf("%i,%i,%i;") for j in parser.iterList("TRIANGLE_SET")]

        parser.endBlock()

    for i in parser.iterList("WEIGHTS"):
        parser.beginBlock()

        wmap = MeshWeightMap()
        lod.weightMaps.append(wmap)
        wmap.name = parser.getString("NAME")

        for j in parser.iterList("WEIGHT_SET"):
            parser.beginBlock()

            vtx, weight = parser.scanf("%i;%f;")
            wmap.values[vtx] = weight

            parser.endBlock()

        parser.endBlock()

    for i in parser.iterList("MORPHS"):
        parser.beginBlock()

        mmap = MeshMorphMap()
        lod.morphMaps.append(mmap)
        mmap.name = parser.getString("NAME")
        mmap.relative = parser.getBool("RELATIVE")

        for j in parser.iterList("MORPH_SET"):
            parser.beginBlock()

            values = parser.scanf("%i;%f,%f,%f;%f,%f,%f;")

            mmap.values[values[0]] = values[1:]

            parser.endBlock()

        parser.endBlock()

    parser.expect("SE_MESH_END")

    return lod

def readSkeletonBin(filename: str) -> list:
    lods = []

    with open(filename, "rb") as file:
        magic = file.read(4)
        if (magic != b"SKEL"):
            raise ValueError("Invalid binary skeleton file")

        ver = struct.unpack("i", file.read(4))[0]
        if (ver not in (6, 9)):
            raise ValueError("Invalid binary skeleton file version {}".format(ver))

        if (ver == 9):
            file.read(4)

        lodCount = struct.unpack("i", file.read(4))[0]

        if (ver == 9):
            lodCount = lodCount // 5

        for i in range(lodCount):

            lod = SkeletonLOD()
            lods.append(lod)

            readString(file)

            lod.maxDist = struct.unpack("f", file.read(4))[0]

            boneCount = struct.unpack("i", file.read(4))[0]
            for j in range(boneCount):
                bone = SkeletonBone()
                lod.bones.append(bone)

                bone.name = readString(file)
                bone.parentName = readString(file)

                bone.absPlacement[0] = struct.unpack("4f", file.read(16))
                bone.absPlacement[1] = struct.unpack("4f", file.read(16))
                bone.absPlacement[2] = struct.unpack("4f", file.read(16))

                bone.relPlacement[0] = struct.unpack("3f", file.read(12))
                bone.relPlacement[1] = struct.unpack("4f", file.read(16))

                bone.offsetLen = struct.unpack("f", file.read(4))[0]
                bone.length = struct.unpack("f", file.read(4))[0]

            lod.sortBones()

    return lods

def readSkeletonAscii(filename: str) -> SkeletonLOD:
    lod = SkeletonLOD()

    boneMap = dict()
    with open(filename, "r") as file:
        parser = asciiparser.AsciiParser(StringIO(file.read()))

    ver = parser.getFloat("SE_SKELETON")
    if (ver != 0.1):
        raise ValueError("Invalid ascii skeleton file version {}".format(ver))

    for i in parser.iterList("BONES"):
        sb = SkeletonBone()
        lod.bones.append(sb)

        sb.name = parser.getString("NAME")
        sb.parentName = parser.getString("PARENT")
        sb.length = parser.getFloat("LENGTH")

        parser.beginBlock()

        values = parser.scanf("%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f;")
        sb.absPlacement = [ values[0:4], values[4:8], values[8:12] ]    # RELATIVE matrix

        parser.endBlock()

        boneMap[sb.name] = sb

    parser.expect("SE_SKELETON_END")

    lod.sortBones()

    # convert child bones positions to absolute
    for sb in lod.bones:
        if (sb.parentName):
            parentBone = boneMap[sb.parentName]

            parentMat = Matrix(parentBone.absPlacement+[(0, 0, 0, 1)])
            mat = Matrix(sb.absPlacement+[(0, 0, 0, 1)])

            mat = parentMat @ mat

            sb.absPlacement = [tuple(mat[0]), tuple(mat[1]), tuple(mat[2])]

    return lod

def writeSkeletonBin(skeletonlods: list, filename: str, ver: int):
    with open(filename, "wb") as file:
        file.write(b"SKEL")
        file.write(struct.pack("i", ver))

        if (ver > 6):
            fileSizePos = file.tell()
            file.write(b"\x00\x00\x00\x00")

        lodCount = len(skeletonlods)
        if (ver > 6):
            lodCount *= 5

        file.write(struct.pack("i", lodCount))
        for lod in skeletonlods:
            writeString(file, "")   # no source filename
            file.write(struct.pack("f", lod.maxDist))

            file.write(struct.pack("i", len(lod.bones)))
            for bone in lod.bones:
                writeString(file, bone.name)
                writeString(file, bone.parentName)

                file.write(struct.pack("12f", *bone.absPlacement[0], *bone.absPlacement[1], *bone.absPlacement[2]))

                file.write(struct.pack("7f", *bone.relPlacement[0], *bone.relPlacement[1]))

                file.write(struct.pack("f", bone.offsetLen))
                file.write(struct.pack("f", bone.length))

        if (ver > 6):
            fileSize = file.tell()
            file.seek(fileSizePos)
            file.write(struct.pack("I", fileSize))

def readAnimsetBin(filename: str) -> list:
    anims = []

    with open(filename, "rb") as file:
        magic = file.read(4)
        if (magic != b"ANIM"):
            raise ValueError("Invalid binary animation set file")

        ver = struct.unpack("i", file.read(4))[0]
        if (ver not in (14, 20, 21)):
            raise ValueError("Invalid binary animation set file version {}".format(ver))

        ainmCount = struct.unpack("i", file.read(4))[0]

        if (ver == 20):
            ainmCount >>= 5
            file.read(4)

        for i in range(ainmCount):
            anim = Animation()
            anims.append(anim)

            readString(file)
            anim.name = readString(file)

            anim.secPerFrame = struct.unpack("f", file.read(4))[0]
            anim.frameCount = struct.unpack("i", file.read(4))[0]
            anim.treshold = struct.unpack("f", file.read(4))[0]

            file.seek(8, 1) # compressed + custom speed

            boneEvpCount = struct.unpack("i", file.read(4))[0]
            for j in range(boneEvpCount):
                evp = BoneEnvelope()
                anim.boneEvps.append(evp)

                evp.boneName = readString(file)

                evp.defPos[0] = struct.unpack("4f", file.read(16))
                evp.defPos[1] = struct.unpack("4f", file.read(16))
                evp.defPos[2] = struct.unpack("4f", file.read(16))

                posCount = struct.unpack("i", file.read(4))[0]
                for k in range(posCount):
                    frameNo = struct.unpack("H2x", file.read(4))[0]
                    evp.positions[frameNo] = struct.unpack("3f", file.read(12))

                rotCount = struct.unpack("i", file.read(4))[0]
                for k in range(rotCount):
                    frameNo = struct.unpack("H2x", file.read(4))[0]
                    evp.rotations[frameNo] = struct.unpack("4f", file.read(16))

                evp.offsetLen = struct.unpack("f", file.read(4))[0]

            morphEvpCount = struct.unpack("i", file.read(4))[0]
            for j in range(morphEvpCount):
                evp = MorphEnvelope()
                anim.morphEvps.append(evp)

                evp.mapName = readString(file)

                factorCount = struct.unpack("i", file.read(4))[0]
                if (factorCount):
                    evp.factors = struct.unpack("{}f".format(factorCount), file.read(4*factorCount))

    return anims

def writeAnimsetBin(anims: list, filename: str, ver: int):
    with open(filename, "wb") as file:
        file.write(b"ANIM")
        file.write(struct.pack("i", ver))

        ainmCount = len(anims)
        if (ver == 20):
            ainmCount <<= 5

            fileSizePos = file.tell()
            file.write(b"\x00\x00\x00\x00")

        file.write(struct.pack("i", ainmCount))
        for anim in anims:
            writeString(file, "")   # no source filename

            writeString(file, anim.name)

            file.write(struct.pack("f", anim.secPerFrame))
            file.write(struct.pack("i", anim.frameCount))
            file.write(struct.pack("f", anim.treshold))

            file.write(struct.pack("i", 0)) # no compression
            file.write(struct.pack("i", 0)) # no custom speed

            file.write(struct.pack("i", len(anim.boneEvps)))
            for evp in anim.boneEvps:
                writeString(file, evp.boneName)

                file.write(struct.pack("12f", *evp.defPos[0], *evp.defPos[1], *evp.defPos[2]))

                file.write(struct.pack("i", len(evp.positions)))
                for frm, pos in evp.positions.items():
                    file.write(struct.pack("H2x3f", frm, *pos))

                file.write(struct.pack("i", len(evp.rotations)))
                for frm, rot in evp.rotations.items():
                    file.write(struct.pack("H2x4f", frm, *rot))

                file.write(struct.pack("f", evp.offsetLen))

            file.write(struct.pack("i", len(anim.morphEvps)))
            for evp in anim.morphEvps:
                writeString(file, evp.mapName)

                factorCount = len(evp.factors)
                file.write(struct.pack("i", factorCount))
                if (evp.factors):
                    file.write(struct.pack("{}f".format(factorCount), *evp.factors))

        if (ver == 20):
            fileSize = file.tell()
            file.seek(fileSizePos)
            file.write(struct.pack("I", fileSize))

def readModelConfigurationBin(filename: str) -> list:   # [[meshFNs], [skeletonFNs], [animsetFNs]]
    meshes = []
    skeletons = []
    animsets = []

    with open(filename, "rb") as file:
        magic = file.read(4)
        if (magic != b"MI03"):
            raise ValueError("Invalid binary model configuration file")

        # source filename
        magic = file.read(4)
        if (magic != b"MISF"):
            raise ValueError("Not a model configuration source filename")

        readBinFileName(file)

        # model name
        readString(file)

        # unk
        struct.unpack("i", file.read(4))[0]

        # stretch?
        struct.unpack("3f", file.read(12))

        # unk
        struct.unpack("i", file.read(4))[0]

        # meshes
        magic = file.read(4)
        if (magic != b"MSHI"):
            raise ValueError("Not a meshes")

        meshCount = struct.unpack("i", file.read(4))[0]
        for i in range(meshCount):
            magic = file.read(4)
            if (magic != b"MESH"):
                raise ValueError("Not a mesh")

            fn = readBinFileName(file)
            meshes.append(fn)

            # textures
            magic = file.read(4)
            if (magic != b"MITS"):
                raise ValueError("Not a model textures")

            texCount = struct.unpack("i", file.read(4))[0]
            for i in range(texCount):
                magic = file.read(4)
                if (magic != b"TITX"):
                    raise ValueError("Not a texture")

                # filename
                readBinFileName(file)
                # name
                readString(file)

        # skeletons
        magic = file.read(4)
        if (magic != b"SKEL"):
            raise ValueError("Not a skeletons")

        skeletonCount = struct.unpack("i", file.read(4))[0]
        for i in range(skeletonCount):
            fn = readBinFileName(file)
            skeletons.append(fn)

        # animsets
        magic = file.read(4)
        if (magic != b"ANAS"):
            raise ValueError("Not a anim sets")

        animsetCount = struct.unpack("i", file.read(4))[0]
        for i in range(animsetCount):
            fn = readBinFileName(file)
            animsets.append(fn)

    return (meshes, skeletons, animsets)