import string

ILLEGAL_PATTERN_CHARACTERS = "{}\n"

class AsciiParser:
    __slots__ = [
        "_file",
        "_depth",
    ]

    def __init__(self, file):
        """Construct new text parser. `file` must be object of TextIOBase derived class opened for reading"""
        self._file = file
        self._depth = 0

    def getValue(self, end: str=";") -> str:
        """Get string value ended by whitespace, EOF, block bracket or suffix `end`"""
        value = ""

        escape = False

        while (True):
            prevPos = self._file.tell()
            c = self._file.read(1)

            # EOF
            if (not c):
                break

            # toggle escape mode for string literals
            if (c == '"'):
                escape = not escape

            # whitespaces
            if (c in string.whitespace and not escape):
                # whitespace before value - skip it
                if (not value):
                    continue
                # after - value ended
                else:
                    break

            # block brackets
            if (c in "{}" and not escape):
                # block start/end after value - value ended
                if (value):
                    # fall back
                    self._file.seek(prevPos)
                    break
                raise RuntimeError("Expected value but found block")

            # end suffix
            if (end):
                suffix = c
                
                stop = False

                pos = self._file.tell()
                while (end.startswith(suffix)):
                    # full match - value ended
                    if (len(end) == len(suffix)):
                        stop = True
                        break

                    p = self._file.read(1)
                    # EOF
                    if (not p):
                        break

                    suffix += p

                if (stop):
                    break

                self._file.seek(pos)

            value += c

        if (escape):
            raise RuntimeError("Unterminated string literal")

        return value

    def peekValue(self, end: str=";") -> str:
        """Get string value without moving in file"""
        pos = self._file.tell()

        value = self.getValue(end)

        self._file.seek(pos)

        return value

    def expect(self, value: str):
        """Compare next string value with `value`"""
        v = self.getValue()
        if (v != value):
            raise RuntimeError("Expected {} but found {}".format(value, v))

    def getString(self, tag: str="", end: str=";") -> str:
        """Get string literal with optional `tag` and suffix `end`"""
        if (tag):
            self.expect(tag)

        value = self.getValue(end)

        if (value[0] != '"' or value[-1] != '"'):
            raise RuntimeError("Expected string value but found {}".format(value))

        return value[1:-1]

    def getInt(self, tag: str="", end: str=";") -> int:
        """Get integer number with optional `tag` and suffix `end`"""
        if (tag):
            self.expect(tag)

        value = self.getValue(end)

        return int(value)

    def getFloat(self, tag: str="", end: str=";") -> float:
        """Get float number with optional `tag` and suffix `end`"""
        if (tag):
            self.expect(tag)

        value =  self.getValue(end)

        return float(value)

    def getBool(self, tag: str="", end: str=";") -> bool:
        """Get boolean value with optional `tag` and suffix `end`"""
        if (tag):
            self.expect(tag)

        value = self.getValue(end)

        return value == "TRUE"

    def scanf(self, fmt: str) -> tuple:
        """Get values specified in `fmt`, supported patterns: %i - int, %f - float, %s - string, %b - bool"""
        buf = ""
        values = []

        i = 0
        while (i < len(fmt)):
            c = fmt[i]
            i += 1

            # skip whitespaces
            if (c in string.whitespace):
                # flush buffer
                if (buf):
                    self.expect(buf)
                    buf = ""
                continue

            if (c in ILLEGAL_PATTERN_CHARACTERS):
                raise ValueError("Forbidden character '{}' in format".format(c))

            if (c == "%"):
                # flush buffer
                if (buf):
                    self.expect(buf)
                    buf = ""
            elif (buf == "%"):
                suffix = ""
                for j in range(i, len(fmt)):
                    p = fmt[j]

                    if (p in string.whitespace):
                        break
                    if (p == "%"):
                        if (j != len(fmt)-1 and fmt[j+1] != "%"):
                            break

                    suffix += p

                i += len(suffix)

                if (c == "i"):
                    values.append(self.getInt(end=suffix))
                elif (c == "f"):
                    values.append(self.getFloat(end=suffix))
                elif (c == "s"):
                    values.append(self.getString(end=suffix))
                elif (c == "b"):
                    values.append(self.getBool(end=suffix))
                else:
                    raise ValueError("Invalid pattern %{}, supported are %i %f %s %b".format(c))

                buf = ""
                continue

            buf += c

        # flush buffer
        if (buf):
            self.expect(buf)
            buf = ""

        return tuple(values)

    def beginBlock(self):
        """Read opened block bracket '{' and increase depth level"""
        while (True):
            c = self._file.read(1)

            # EOF
            if (not c):
                raise RuntimeError("Expected block start but found EOF")

            if (c == "{"):
                break

            if (c not in string.whitespace):
                raise RuntimeError("Expected block start bound found {}".format(c))

        self._depth += 1

    def endBlock(self):
        """Read closed block bracket '}' and decrease depth level"""
        if (not self._depth):
            raise RuntimeError("Cannot go below 0 depth level")

        while (True):
            c = self._file.read(1)

            # EOF
            if (not c):
                raise RuntimeError("Expected block end but found EOF")

            if (c == "}"):
                break

            if (c not in string.whitespace):
                raise RuntimeError("Expected block end bound found {}".format(c))

        self._depth -= 1

    def skipBlock(self, tag: str=""):
        """Skip block entirely"""
        if (tag):
            self.expect(tag)

        startDepth = self._depth

        self.beginBlock()

        curDepth = self._depth

        while (True):
            prevPos = self._file.tell()
            c = self._file.read(1)

            # EOF
            if (not c):
                raise RuntimeError("Could not find block end - EOF reached")

            # inner blocks - update depth
            if (c == "{"):
                curDepth += 1
            elif (c == "}"):
                curDepth -= 1

            # same depth - block ended
            if (curDepth == startDepth):
                self._file.seek(prevPos)
                break

        self.endBlock()

    def iterList(self, tag: str=""):
        """Check optional `tag`, read element count and begin a block. Yields range(0, elementCount). List block must be manually handled so block successfuly ended"""
        count = self.getInt(tag, "")

        curDepth = self._depth
        self.beginBlock()

        for i in range(count):
            yield i

        self.endBlock()

        if (self._depth != curDepth):
            raise RuntimeError("Inconsistent depth level: before list - {}, after - {}".format(curDepth, self._depth))

    def iterBlock(self, tag: str=""):
        """Same as iterList but when element count not specified and deduced from brackets"""
        if (tag):
            self.expect(tag)

        curDepth = self._depth

        self.beginBlock()

        index = 0

        while (True):
            # check if block about to end
            pos = self._file.tell()
            end = False
            while (True):
                c = self._file.read(1)

                # EOF
                if (not c):
                    break

                # end of block
                if (c == "}"):
                    end = True
                    break

                # some text - continue iteration
                if (c not in string.whitespace):
                    break
            self._file.seek(pos)

            # about to end current block
            if (end and self._depth - 1 <= curDepth):
                break

            yield index

            index += 1

        self.endBlock()

class AsciiEncoder:
    __slots__ = [
        "_file",
        "_depth",
    ]

    def __init__(self, file):
        """Construct new text encoder. `file` must be object of TextIOBase derived class opened for writing"""
        self._file = file
        self._depth = 0

    def putValue(self, value: str, tag: str="", end: str=";", indent: bool=True, newline: bool=True):
        """Put string value with optional `tag`, suffix `end`, depth-level identaion and new line character"""
        text = ""

        if (indent):
            text = "\t" * self._depth

        if (tag):
            text += "{} ".format(tag)

        text += value

        if (end):
            text += end

        if (newline):
            text += "\n"


        self._file.write(text)

    def putString(self, value: str, tag: str="", end: str=";", indent: bool=True, newline: bool=True):
        """Put string literal"""
        self.putValue('"{}"'.format(value), tag, end, indent, newline)

    def putNumber(self, value, tag: str="", end: str=";", indent: bool=True, newline: bool=True):
        """Put number (integer or float)"""
        self.putValue(str(value), tag, end, indent, newline)

    def putBool(self, value: bool, tag: str="", end: str=";", indent: bool=True, newline: bool=True):
        """Put boolean value"""
        self.putValue("TRUE" if value else "FALSE", tag, end, indent, newline)

    def printf(self, fmt: str, *values, **kw):
        """Put values specified in `fmt`. Keyword arguments: `indent` - enable identation, `newline` - insert tailing new line character"""
        if (not fmt):
            return

        if (kw.get("indent", True)):
            self._file.write("\t" * self._depth)

        buf = ""
        idx = 0
        for c in fmt:
            if (c in ILLEGAL_PATTERN_CHARACTERS):
                raise ValueError("Forbidden character '{}' in format".format(c))

            if (c == "%"):
                # flush buffer
                if (buf):
                    self._file.write(buf)
                    buf = ""
            elif (buf == "%"):
                if (idx > len(values)):
                    raise ValueError("Not enough values for given format")
                v = values[idx]
                idx += 1

                if (c == "i"):
                    self._file.write(str(int(v)))
                elif (c == "f"):
                    self._file.write(str(float(v)))
                elif (c == "s"):
                    self._file.write('"{}"'.format(v))
                elif (c == "b"):
                    self._file.write("TRUE" if bool(v) else "FALSE")
                else:
                    raise ValueError("Invalid pattern %{}, supported are %i %f %s %b".format(c))

                buf = ""
                continue

            buf += c

        # flush buffer
        if (buf):
            self._file.write(buf)
            buf = ""

        if (kw.get("newline", True)):
            self._file.write("\n")

    def beginBlock(self, indent: bool=True, newline: bool=True):
        """Write open block bracket '{' with optional identation and tailing new line character and increase depth level"""
        self.putValue("{", end="", indent=indent, newline=newline)
        self._depth += 1

    def endBlock(self, indent: bool=True, newline: bool=True):
        """Write closed block bracket '}' with optional identaion and tailing new line character and decrease depth level"""
        if (not self._depth):
            raise RuntimeError("Cannot go below 0 depth level")

        self._depth -=1
        self.putValue("}", end="", indent=indent, newline=newline)

    def iterList(self, iterable, tag: str="", writeCount: bool=True, indent: bool=True, newline: bool=True):
        """Write optional `tag`, element count if `writeCount` enabled, identation and tailing new line character, begin block, iterate elements and finally end block. Yields elements of `iterable`."""
        self.putValue(str(len(iterable)) if writeCount else "", tag, "", indent, newline)

        self.beginBlock(indent, newline)

        for e in iterable:
            yield e

        self.endBlock(indent, newline)