bl_info = {
    "name": "Serious Engine 1 SKA",
    "author": "other1",
    "version": (2, 1),
    "blender": (2, 80, 0),
    "location": "File > Import-Export",
    "description": "Import/export Serious Engine 1 SKA models. Tested on: Serious Sam 1.07, Serious Sam XBOX, Nitro Family, EuroCops, Last Chaos, Rakion Chaos Force",
    "category": "Import-Export",
}

from . import importers, exporters
import bpy, os.path

from bpy.props import (
        StringProperty,
        BoolProperty,
        EnumProperty,
        CollectionProperty,
        )
        
from bpy_extras.io_utils import ImportHelper, ExportHelper, axis_conversion

class IMPORT_OP_SE1_SKA(bpy.types.Operator, ImportHelper):
    """Load a Serious Engine 1 SKA file"""
    bl_idname= "import_scene.se1_ska"
    bl_label= "Import Serious Engine 1 SKA"
    bl_options= {'REGISTER', 'UNDO'}
    
    filter_glob : StringProperty(
        default="*.bmc;*.smc;*.bm;*.bs;*.ba;*.am;*.as;*.aa",
        options={'HIDDEN'},
    )
    
    files: CollectionProperty(
        name="File Path",
        type=bpy.types.OperatorFileListElement,
    )
            
    importMesh : BoolProperty(
        name="Import mesh (for .smc/.bmc)",
        description="Imports mesh of .smc file", 
        default=True
    )

    importSkeleton : BoolProperty(
        name="Import skeleton (for .smc/.bmc)", 
        description="Imports skeleton of .smc file", 
        default=True
    )

    importAnimations : BoolProperty(
        name="Import animations (for .smc/.bmc)", 
        description="Imports animations of .smc file", 
        default=True
    )   

    applyArmature : BoolProperty(
        name="Apply armature (for .smc/.bmc)", 
        description="Apply armatures to meshes", 
        default=True
    )

    removeDoubles : BoolProperty(name="Merge vertices (for meshes)", 
        description="Merge near vertices", 
        default=True
    )

    forwardAxis : EnumProperty(
            items=(("X", "X", "Forward is positive X axis"),
                   ("Y", "Y", "Forward is positive Y axis"),
                   ("Z", "Z", "Forward is positive Z axis"),
                   ("-X", "-X", "Forward is negative X axis"),
                   ("-Y", "-Y", "Forward is negative Y axis"),
                   ("-Z", "-Z", "Forward is negative Z axis")
                   ),
            name="Forward axis",
            description="Source forward axis",
            default="-Z"
    )

    upAxis : EnumProperty(
            items=(("X", "X", "Up is positive X axis"),
                   ("Y", "Y", "Up is positive Y axis"),
                   ("Z", "Z", "Up is positive Z axis"),
                   ("-X", "-X", "Up is negative X axis"),
                   ("-Y", "-Y", "Up is negative Y axis"),
                   ("-Z", "-Z", "Up is negative Z axis")
                   ),
            name="Up axis",
            description="Source Up axis",
            default="Y"
    )

    def execute(self, context):
        context.window.cursor_modal_set("WAIT")

        convMat = axis_conversion(self.forwardAxis, self.upAxis).to_4x4()

        try:
            fns = []
            meshes = []
            skeletons = []
            animations = []
            if (self.files):
                dirname = os.path.dirname(self.filepath)
                for file in self.files:
                    path = os.path.join(dirname, file.name)
                    if (path.casefold().endswith("m")):
                        meshes.append(path)
                    elif (path.casefold().endswith("s")):
                        skeletons.append(path)
                    elif (path.casefold().endswith("a")):
                        animations.append(path)
                    else:
                        fns.append(path)
                fns += meshes + skeletons + animations
            else:
                fns = [self.filepath]

            importers.load(fns, self.importMesh, self.importSkeleton, self.importAnimations, self.applyArmature, self.removeDoubles, convMat)
        except Exception:
            context.window.cursor_modal_restore()
            raise

        context.window.cursor_modal_restore()
        return {'FINISHED'}

class EXPORT_OP_SE1_SKA(bpy.types.Operator, ExportHelper):
    """Export scene to Serious Engine 1 SKA format"""
    bl_idname = "export_scene.se1_ska"
    bl_label = "Export SE1 SKA"

    filename_ext = ""

    filter_glob : StringProperty(
        default="*.am;*.as;*.aa;*.bm;*.bs;*.ba;*.smc",
        options={'HIDDEN'},
    )

    formatType : EnumProperty(
        items=(("BIN", "Binary", "Export in SKA binary format"),
                ("ASCII", "Ascii", "Export in SKA Ascii format"),
                ),
        name="Format",
        description="SKA format options"
    )

    version : EnumProperty(
        items=(
            ("1.07", "SE 1.07", "Serious Engine 1.07"),
            (">1.07", "SE > 1.07", "Serious Engine above 1.07"),
            ("LC", "Last Chaos", "Serious Engine of Last Chaos"),
        ),
        name="Version",
        description="SKA version"
    )

    meshType : EnumProperty(
        items=(("0", "No", "Export as normal mesh"),
               ("1", "Half face forward", "Export as half face forward mesh"),
               ("2", "Full face forward", "Export as full face forward mesh"),
               ),
        name="Face forward",
        description="Mesh face forward options",
    )

    exportMesh : BoolProperty(
        name="Mesh",
        description="Export mesh objects",
        default=True,
    )

    exportArmature : BoolProperty(
        name="Armature",
        description="Export armature objects",
        default=True,
    )

    exportAnimation : BoolProperty(
        name="Animation",
        description="Export animations",
        default=True,
    )

    selectionOnly : BoolProperty(
        name="Selection",
        description="Export only selected objects",
        default=False,
    )

    makeConfiguration : BoolProperty(
        name="Make model configuration (.smc)",
        description="Make model configuration file for binary format",
        default=True,
    )

    handleMeshes : EnumProperty(
        items=(("MESH", "As meshes", "Export each mesh object as mesh"),
                 ("LOD", "As LODs", "Export each mesh object as mesh LOD")
              ),
        name="Handle meshes",
        description="Export type of multiple mesh objects",
        default="MESH"
    )

    handleArmatures : EnumProperty(
        items=(("SKELETON", "As skeletons", "Export each armature object as skeleton"),
                 ("LOD", "As LODs", "Export each armature object as skeleton LOD")
              ),
        name="Handle armatures",
        description="Export type of multiple armature objects",
        default="SKELETON"
    )

    forwardAxis : EnumProperty(
            items=(("X", "X", "Forward is positive X axis"),
                   ("Y", "Y", "Forward is positive Y axis"),
                   ("Z", "Z", "Forward is positive Z axis"),
                   ("-X", "-X", "Forward is negative X axis"),
                   ("-Y", "-Y", "Forward is negative Y axis"),
                   ("-Z", "-Z", "Forward is negative Z axis")
                   ),
            name="Forward axis",
            description="Export forward axis",
            default="-Z"
    )

    upAxis : EnumProperty(
            items=(("X", "X", "Up is positive X axis"),
                   ("Y", "Y", "Up is positive Y axis"),
                   ("Z", "Z", "Up is positive Z axis"),
                   ("-X", "-X", "Up is negative X axis"),
                   ("-Y", "-Y", "Up is negative Y axis"),
                   ("-Z", "-Z", "Up is negative Z axis")
                   ),
            name="Up axis",
            description="Export Up axis",
            default="Y"
    )

    def execute(self, context):
        context.window.cursor_modal_set("WAIT")

        try:
            exporters.save(self.filepath, self.formatType == "BIN", self.makeConfiguration, self.exportMesh, self.exportArmature, self.exportAnimation, int(self.meshType), self.selectionOnly, self.forwardAxis, self.upAxis, self.version, self.handleMeshes, self.handleArmatures)
        except Exception:
            context.window.cursor_modal_restore()
            raise

        context.window.cursor_modal_restore()

        return {'FINISHED'}

def menu_func_import(self, context):
    self.layout.operator(IMPORT_OP_SE1_SKA.bl_idname, text="Serious Engine 1 SKA")

def menu_func_export(self, context):
    self.layout.operator(EXPORT_OP_SE1_SKA.bl_idname, text="Serious Engine 1 SKA")

def register():
    bpy.utils.register_class(IMPORT_OP_SE1_SKA)
    bpy.utils.register_class(EXPORT_OP_SE1_SKA)

    bpy.types.TOPBAR_MT_file_import.append(menu_func_import)
    bpy.types.TOPBAR_MT_file_export.append(menu_func_export)

def unregister():
    bpy.utils.unregister_class(IMPORT_OP_SE1_SKA)
    bpy.utils.unregister_class(EXPORT_OP_SE1_SKA)

    bpy.types.TOPBAR_MT_file_import.remove(menu_func_import)
    bpy.types.TOPBAR_MT_file_export.remove(menu_func_export)

if __name__ == "__main__":
    register()
