from mathutils import Matrix, Quaternion, Vector
from bpy_extras.io_utils import axis_conversion
import bpy, bmesh, os

from . import ska
from . import asciiparser
from . import utils

class Loop:
    __slots__ = [
        "loop",
        "newIndex",
        "uvLayers",
        "deformLayer",
        "shapeLayers",
    ]

    def __init__(self, loop, uvLayers, deformLayer, shapeLayers):
        self.loop = loop
        self.newIndex = 0

        self.uvLayers = uvLayers
        self.deformLayer = deformLayer
        self.shapeLayers = shapeLayers

    def __eq__(self, other):
        return self.compare(other) == 0

    def __lt__(self, other):
        return self.compare(other) < 0

    def compare(self, other):
        if (self.material_index() != other.material_index()):
            return self.material_index() - other.material_index()

        for selfCo, otherCo in zip(self.co(), other.co()):
            if (selfCo != otherCo):
                return selfCo - otherCo

        for selfCo, otherCo in zip(self.normal(), other.normal()):
            if (selfCo != otherCo):
                return selfCo - otherCo

        for selfUV, otherUV in zip(self.uvs(), other.uvs()):
            for selfCo, otherCo in zip(selfUV, otherUV):
                if (selfCo != otherCo):
                    return selfCo - otherCo

        selfWeights = self.weights()
        otherWeights = other.weights()

        selfShapes = self.shapes()
        otherShapes = other.shapes()

        if (len(selfWeights) != len(otherWeights)):
            return selfWeights - otherWeights

        for selfW, otherW in zip(selfWeights, otherWeights):
            for selfValue, otherValue in zip(selfW, otherW):
                if (selfValue != otherValue):
                    return selfValue - otherValue

        for selfS, otherS in zip(selfShapes, otherShapes):
            if (selfS != otherS):
                return selfS - otherS

        return 0

    def co(self):
        return self.loop.vert.co

    def normal(self):
        return self.loop.vert.normal

    def uvs(self) -> list:
        uvs = []
        for i in range(len(self.uvLayers)):
            uvs.append(self.loop[self.uvLayers[i]].uv)
        return uvs

    def material_index(self) -> int:
        return self.loop.face.material_index

    def weights(self) -> list:
        weights = []

        if (len(self.deformLayer)):
            for idx, w in self.loop.vert[self.deformLayer[0]].items():
                weights.append([idx, w])

        return weights

    def shapes(self) -> list:
        shapes = []

        for i in range(len(self.shapeLayers)):
            shapes.append(self.loop.vert[self.shapeLayers[i]])

        return shapes

def toSkaMesh(objects: list, convMat: Matrix) -> list:
    lods = []

    mipStep = 15.0
    mip = mipStep

    for obj in objects:
        if (obj.type != "MESH"):
            continue
        bpy.context.view_layer.objects.active = obj

        lod = ska.MeshLOD()
        lods.append(lod)

        lod.maxDist = mip
        lod.flags = 0

        mip += mipStep

        # collect shape key normals if has any
        skNormals = []
        if (obj.data.shape_keys):
            bpy.context.object.active_shape_key_index = 0
            for kb in obj.data.shape_keys.key_blocks[1:]:
                skNormals.append(kb.normals_vertex_get())

        # create empty uv maps
        for uvl in obj.data.uv_layers:
            uvmap = ska.MeshUVMap()
            lod.uvmaps.append(uvmap)

            uvmap.name = uvl.name

        bpy.ops.object.mode_set(mode="EDIT")

        bm = bmesh.from_edit_mesh(obj.data)
        bm.verts.ensure_lookup_table()
        bm.faces.ensure_lookup_table()

        deformLayer = None

        if (obj.vertex_groups):
            deformLayer = bm.verts.layers.deform[0]

        # create empty morph maps
        if (obj.data.shape_keys):
            relative = obj.data.shape_keys.use_relative

            for j in range(len(obj.data.shape_keys.key_blocks)):
                l = bm.verts.layers.shape[j]
                if (j):
                    mmap = ska.MeshMorphMap()
                    lod.morphMaps.append(mmap)

                    mmap.name = l.name
                    mmap.relative = relative

        # create empty weight maps
        for vg in obj.vertex_groups:
            wmap = ska.MeshWeightMap()
            wmap.name = vg.name

            lod.weightMaps.append(wmap)

        # collect loops
        loops = []
        loopDict = dict()

        for v in bm.verts:
            for l in v.link_loops:
                loop = Loop(l, bm.loops.layers.uv, bm.verts.layers.deform, bm.verts.layers.shape)
                loops.append(loop)
                loopDict[l] = loop

        # optimize loops
        sortedLoops = sorted(loops)

        optiLoops = []

        lastLoop = sortedLoops[0]
        lastLoop.newIndex = 0
        diffLoops = 1
        optiLoops.append(lastLoop)

        for loop in sortedLoops[1:]:
            if (loop != lastLoop):
                diffLoops += 1
                optiLoops.append(loop)
                lastLoop = loop
            loop.newIndex = diffLoops - 1

        vertexWeights = []

        # collect vertex data
        for loop in optiLoops:
            co = loop.co()
            lod.vertices.append(convMat @ co)

            normal = loop.normal()
            lod.normals.append(convMat @ normal)

            for i, uv in enumerate(loop.uvs()):
                lod.uvmaps[i].coords.append((uv[0], 1.0 - uv[1]))

            weights = loop.weights()

            wmapIndices = []
            wmapWeights = []

            if (weights):
                # validate weights
                totalWeight = 0.0
                for idx, w in weights:
                    totalWeight += w

                if (totalWeight != 1.0):
                    if (len(weights) == 1):
                        weights[0][1] = 1.0
                    else:
                        m = max(weights, key=lambda w: w[1])[1]
                        props = [w[1]/m for w in weights]
                        x = 1.0/sum(props)
                        res = [p*x for p in props]

                        for j, pair in enumerate(weights):
                            pair[1] = res[j]

                for idx, w in weights:
                    lod.weightMaps[idx].values[loop.newIndex] = w
                    wmapIndices.append(idx)
                    wmapWeights.append(w)

            vertexWeights.append((wmapIndices, wmapWeights))

            shapes = loop.shapes()
            if (shapes):
                basis = shapes[0]
                for i, co in enumerate(shapes[1:]):
                    if (basis != co):
                        normal = skNormals[i][loop.loop.vert.index*3:loop.loop.vert.index*3+3]
                        co = convMat @ co
                        normal = convMat @ normal
                        lod.morphMaps[i].values[loop.newIndex] = tuple(co) + tuple(normal)

        # create empty surfaces
        matFaces = []
        for mat in obj.data.materials:
            surf = ska.MeshSurface()
            lod.surfaces.append(surf)

            surf.name = mat.name

            matFaces.append([])

        # separate faces by materials (surfaces)
        for f in bm.faces:
            matFaces[f.material_index].append(f)

        for i, faces in enumerate(matFaces):

            surf = lod.surfaces[i]
            surf.firstVtx = None

            uniqueVerts = set()
            uniqueWeightMaps = set()

            for f in faces:
                tri = []
                for l in f.loops:
                    loop = loopDict[l]
                    idx = loop.newIndex

                    if (surf.firstVtx == None):
                        surf.firstVtx = idx
                    else:
                        surf.firstVtx = min(surf.firstVtx, idx)

                    uniqueVerts.add(idx)
                    tri.append(idx)

                    for idx, w in loop.weights():
                        uniqueWeightMaps.add(idx)

                if (len(tri) != 3):
                    raise RuntimeError("All polygons must be triangles")

                surf.tris.append(tri)

            surf.tris = [list(map(lambda idx: idx-surf.firstVtx, tri)) for tri in surf.tris]

            surf.vtxCount = len(uniqueVerts)

            surf.weightMaps = list(uniqueWeightMaps)

        # create vertex weights
        for i, (indices, weights) in enumerate(vertexWeights):
            wmapIndices = [0, 0, 0, 0]
            wmapWeights = [0, 0, 0, 0]

            loop = optiLoops[i]
            surf = lod.surfaces[loop.material_index()]

            for j, (idx, w) in enumerate(zip(indices, weights)):
                local_idx = surf.weightMaps.index(idx)
                wmapIndices[j] = local_idx
                # wmapWeights[len(weights) - 1 - j] = min(max(int(w * 255), 0), 255)
                wmapWeights[j] = min(max(int(w * 255), 0), 255)

            lod.vertexWeights.append((wmapIndices, wmapWeights))

        bpy.ops.object.mode_set(mode="OBJECT")

    return lods

def toSkaSkeleton(objects: list, convMat: Matrix, convMatInv: Matrix) -> list:
    lods = []

    mipStep = 15.0
    mip = mipStep

    for obj in objects:
        if (obj.type != "ARMATURE"):
            continue

        bpy.context.view_layer.objects.active = obj

        lod = ska.SkeletonLOD()
        lods.append(lod)

        lod.maxDist = mip
        mip += mipStep

        bpy.ops.object.mode_set(mode="EDIT")

        # collect bones
        arm = obj.data
        for b in arm.edit_bones:
            sb = ska.SkeletonBone()
            lod.bones.append(sb)
            sb.name = b.name

            absMat = obj.matrix_world @ b.matrix @ obj.matrix_world.inverted()
            absMat = convMat @ absMat @ convMatInv

            if (b.parent):
                sb.parentName = b.parent.name
                relMat = (b.parent.matrix.inverted() @ b.matrix)
                relMat = obj.matrix_world @ relMat @ obj.matrix_world.inverted()
                relMat = convMat @ relMat @ convMatInv
            else:
                relMat = absMat

            sb.absPlacement = [ tuple(absMat[0]), tuple(absMat[1]), tuple(absMat[2]) ]

            relQuat = relMat.to_quaternion()
            relPos = relMat.to_translation()
            sb.relPlacement = [ tuple(relPos), tuple(relQuat) ]
            sb.offsetLen = relPos.length

            sb.length = b.length

        bpy.ops.object.mode_set(mode="OBJECT")

        lod.sortBones()

    return lods

def toSkaAnimset(armObj, convMat: Matrix, convMatInv: Matrix) -> list:
    anims = []

    if (armObj):
        bpy.context.view_layer.objects.active = armObj
        bpy.ops.object.mode_set(mode="POSE")
        if (not armObj.animation_data):
            armObj.animation_data_create()

    for action in bpy.data.actions:
        name = action.name
        shapeAction = False
        # if action name starts with SK_ there must be corresponding armature action 
        if (action.name.startswith("SK_")):
            armAction = bpy.data.actions.get(action.name[3:], None)
            if (not armAction):
                # if not found - collect only morph envelopes for this animation
                name = action.name[3:]
                shapeAction = True
            else:
                # do nothing - this action will be handled by armature action
                continue

        anim = ska.Animation()
        anims.append(anim)
        anim.name = action.name
        anim.secPerFrame = 1/30

        firstFrame = int(action.frame_range[0])
        lastFrame = int(action.frame_range[1])

        anim.frameCount = lastFrame - firstFrame + 1

        if (shapeAction):
            # shape key-only action
            skaction = action
        else:
            # lookup for corresponding shape key action
            skaction = bpy.data.actions.get("SK_{}".format(action.name), None)

        if (skaction):
            # create morph envelopes
            for fc in skaction.fcurves:
                if (fc.data_path.startswith("key_blocks") and fc.keyframe_points):
                    evp = ska.MorphEnvelope()
                    anim.morphEvps.append(evp)

                    evp.mapName = fc.data_path[12:fc.data_path.find('"', 12)]

                    evp.factors = [None for i in range(anim.frameCount)]
                    for kp in fc.keyframe_points:
                        evp.factors[int(kp.co[0])] = kp.co[1]


                    # morph envelopes has exactly animation frame count keyframes
                    # so we maybe need to sample them
                    prev = None
                    for i in range(anim.frameCount):
                        f = evp.factors[i]
                        # if has keyframe with None factor
                        if (f is None):
                            nextFactor = None
                            idx = -1
                            # look up for next not-None factor
                            for j, f2 in enumerate(evp.factors[i+1:]):
                                if (not f2 is None):
                                    nextFactor = f2
                                    idx = i + j + 1
                                    break

                            if (not nextFactor is None and prev is None):
                                # if no not-None factors before - fill next value
                                for k in range(idx):
                                    evp.factors[k] = nextFactor
                            elif (nextFactor is None and not prev is None):
                                # if no more not-None factors after - fill previous value
                                for k in range(i, anim.frameCount):
                                    evp.factors[k] = prev
                            else:
                                # sample None values
                                step = 1 / (idx-i+1)
                                lerp = step
                                for k in range(i, idx):
                                    evp.factors[k] = prev + (nextFactor - prev) * lerp
                                    lerp += step
                        prev = f

        # cannot create bone envelopes without armature
        if (not armObj):
            continue

        keyframes = dict()  # {bone: {frm: [hasPos, hasRot], ...}, ...}

        # collect keyframes for bones
        for fc in action.fcurves:
            if (fc.data_path.startswith("pose.bones")):
                boneName = fc.data_path[12:fc.data_path.find('"', 12)]
                bone = armObj.pose.bones.get(boneName, None)
                if (not bone):
                    continue

                dct = keyframes.get(bone, None)
                if (not dct):
                    dct = dict()
                    keyframes[bone] = dct

                    dct[firstFrame] = [True, True]
                    dct[lastFrame] = [True, True]

                if ("location" in fc.data_path):
                    for kp in fc.keyframe_points:
                        frm = int(kp.co[0])
                        frameInfo = dct.get(frm, None)
                        if (not frameInfo):
                            frameInfo = [False, False]
                            dct[frm] = frameInfo
                        frameInfo[0] = True
                elif ("rotation" in fc.data_path):
                    for kp in fc.keyframe_points:
                        frm = int(kp.co[0])
                        frameInfo = dct.get(frm, None)
                        if (not frameInfo):
                            frameInfo = [False, False]
                            dct[frm] = frameInfo
                        frameInfo[1] = True

        armObj.animation_data.action = action

        bpy.ops.pose.transforms_clear()

        evps = dict()
        # create bone envelopes
        for bone, finf in keyframes.items():
            evp = ska.BoneEnvelope()
            anim.boneEvps.append(evp)

            evps[bone] = evp

            evp.boneName = bone.name

            bpy.context.scene.frame_set(0)

            if (bone.parent):
                relMat = convMat @ (armObj.matrix_world @ (bone.parent.bone.matrix_local.inverted() @ bone.bone.matrix_local) @ armObj.matrix_world.inverted()) @ convMatInv
            else:
                relMat = convMat @ (armObj.matrix_world @ bone.bone.matrix_local @ armObj.matrix_world.inverted()) @ convMatInv

            evp.defPos = [tuple(relMat[0]), tuple(relMat[1]), tuple(relMat[2])]
            evp.offsetLen = relMat.translation.length

        # create keyframes
        for j in range(firstFrame, lastFrame + 1):
            bpy.context.scene.frame_set(j)

            for bone, finf in keyframes.items():
                evp = evps[bone]

                info = finf.get(j, [False, False])

                if (not any(info)):
                    continue

                mat = convMat @ (armObj.matrix_world @ (bone.parent.matrix.inverted() @ bone.matrix) @ armObj.matrix_world.inverted()) @ convMatInv \
                    if bone.parent else convMat @ (armObj.matrix_world @ bone.matrix @ armObj.matrix_world.inverted()) @ convMatInv

                if (info[0]):
                    evp.positions[j - firstFrame] = tuple(mat.translation)
                if (info[1]):
                    evp.rotations[j - firstFrame] = tuple(mat.to_quaternion())

        bpy.context.scene.frame_set(0)

    bpy.ops.object.mode_set(mode="OBJECT")

    return anims

def saveMeshBin(filename: str, objects: list, meshtype: int, convMat: Matrix, ver: str):
    if (not objects):
        return

    lods = toSkaMesh(objects, convMat)

    if (meshtype):
        for lod in lods:
            lod.flags = meshtype

    version = {
        "1.07": 12,
        ">1.07": 16,
        "LC": 17,
    }[ver]

    if (version == 17):
        ska.writeMeshBinLC(lods, filename, version)
    else:
        ska.writeMeshBin(lods, filename, version)

def saveMeshAscii(filename: str, objects: list, meshtype: int, convMat: Matrix):
    for ilod, obj in enumerate(objects):
        if (len(objects) == 1):
            fn = filename + ".am"
        else:
            fn = "{}_LOD_{:02}.am".format(filename, ilod)

        with open(fn, "w", encoding="1251") as file:
            encoder = asciiparser.AsciiEncoder(file)
            encoder.putNumber(0.1, "SE_MESH")

            if (meshtype):
                encoder.putBool(True, "HALF_FACE_FORWARD" if meshtype == 1 else "FULL_FACE_FORWARD")

            bpy.context.view_layer.objects.active = obj

            # collect shape keys normals if any
            skNormals = []
            if (obj.data.shape_keys):
                bpy.context.object.active_shape_key_index = 0
                for kb in obj.data.shape_keys.key_blocks[1:]:
                    skNormals.append(kb.normals_vertex_get())

            bpy.ops.object.mode_set(mode="EDIT")

            bm = bmesh.from_edit_mesh(obj.data)
            bm.verts.ensure_lookup_table()
            bm.faces.ensure_lookup_table()

            deformLayer = None
            if (obj.vertex_groups):
                deformLayer = bm.verts.layers.deform[0]
            skLayers = []

            morphMaps = dict()
            relative = True

            # collect shape key layers
            if (obj.data.shape_keys):
                relative = obj.data.shape_keys.use_relative

                for j in range(len(obj.data.shape_keys.key_blocks)):
                    l = bm.verts.layers.shape[j]
                    if (j):
                        morphMaps[l.name] = dict()
                    skLayers.append(l)

            # in ascii format mesh vertices are unique for each polygon
            # instead of collecting vertices we are collecting mesh loops
            loopSet = set()
            for vtx in bm.verts:
                for l in vtx.link_loops:
                    loopSet.add(l)

            loops = sorted(loopSet, key=lambda l: l.index)
            weightMaps = [[vg.name, dict()] for vg in obj.vertex_groups]

            # write vertices
            for l in encoder.iterList(loops, "VERTICES"):
                co = convMat @ (obj.matrix_world @ l.vert.co)
                encoder.printf("%f, %f, %f;", co[0], co[1], co[2])

                # also collect weights if has any
                if (deformLayer):
                    for vgidx, w in l.vert[deformLayer].items():
                        wmap = weightMaps[vgidx][1]
                        wmap[l.index] = w

                # and morphs
                for j, lay in enumerate(skLayers[1:]):
                    basisCo = l.vert[skLayers[0]]
                    co = l.vert[lay]
                    if (co != basisCo):
                        co = convMat @ (obj.matrix_world @ co)
                        normal = convMat @ (obj.matrix_world @ Vector(skNormals[j][l.vert.index*3:l.vert.index*3+3]))
                        morphMaps[lay.name][l.index] = (co[0], co[1], co[2], normal[0], normal[1], normal[2])

            # write normals
            for l in encoder.iterList(loops, "NORMALS"):
                normal = convMat @ (obj.matrix_world @ l.vert.normal)
                encoder.printf("%f, %f, %f;", normal[0], normal[1], normal[2])

            # write uvmaps
            for uvl in encoder.iterList(obj.data.uv_layers, "UVMAPS"):
                encoder.beginBlock()
                layer = bm.loops.layers.uv[uvl.name]

                encoder.putString(uvl.name, "NAME")
                for l in encoder.iterList(loops, "TEXCOORDS"):
                    uv = l[layer].uv
                    encoder.printf("%f, %f;", uv[0], 1.0 - uv[1])

                encoder.endBlock()

            # separate faces by materials (surfaces)
            surfaces = [[mat.name, []] for mat in obj.data.materials]
            for f in bm.faces:
                surfaces[f.material_index][1].append(f)

            surfaces = list(filter(lambda e: e[1], surfaces))

            # write surfaces
            for name, faces in encoder.iterList(surfaces, "SURFACES"):
                encoder.beginBlock()

                encoder.putString(name, "NAME")
                for f in encoder.iterList(faces, "TRIANGLE_SET"):
                    if (len(f.loops) != 3):
                        raise RuntimeError("All faces must be triangles")
                    encoder.printf("%i, %i, %i;", f.loops[0].index, f.loops[1].index, f.loops[2].index)

                encoder.endBlock()

            weightMaps = list(filter(lambda e: e[1], weightMaps))

            # write weights
            for name, weights in encoder.iterList(weightMaps, "WEIGHTS"):
                encoder.beginBlock()

                encoder.putString(name, "NAME")
                for vtx in encoder.iterList(weights, "WEIGHT_SET"):
                    encoder.beginBlock(newline=False)
                    encoder.printf("%i; %f;", vtx, weights[vtx], indent=False, newline=False)
                    encoder.endBlock(indent=False)

                encoder.endBlock()

            for k in list(morphMaps.keys()):
                if (not morphMaps[k]):
                    morphMaps.pop(k)

            # write morphs
            for name in encoder.iterList(morphMaps, "MORPHS"):
                encoder.beginBlock()

                mmap = morphMaps[name]

                encoder.putString(name, "NAME")
                encoder.putBool(relative, "RELATIVE")
                for vtx in encoder.iterList(mmap, "MORPH_SET"):
                    encoder.beginBlock(newline=False)
                    encoder.printf("%i; %f, %f, %f; %f, %f, %f;", vtx, *mmap[vtx], indent=False, newline=False)
                    encoder.endBlock(indent=False)

                encoder.endBlock()

            bpy.ops.object.mode_set(mode="OBJECT")

            encoder.putValue("SE_MESH_END")

def saveSkeletonBin(filename: str, objects: list, convMat: Matrix, convMatInv: Matrix, ver: str):
    if (not objects):
        return

    lods = toSkaSkeleton(objects, convMat, convMatInv)

    version = {
        "1.07": 6,
        ">1.07": 6,
        "LC": 6,
    }[ver]

    ska.writeSkeletonBin(lods, filename, version)

def saveSkeletonAscii(filename: str, objects: list, convMat: Matrix, convMatInv: Matrix):
    for ilod, obj in enumerate(objects):
        if (len(objects) == 1):
            fn = filename + ".as"
        else:
            fn = "{}_LOD_{:02}.as".format(filename, ilod)

        with open(fn, "w") as file:
            encoder = asciiparser.AsciiEncoder(file)
            encoder.putNumber(0.1, "SE_SKELETON")

            bpy.context.view_layer.objects.active = obj

            bpy.ops.object.mode_set(mode="EDIT")

            arm = obj.data

            for b in encoder.iterList(arm.edit_bones, "BONES"):
                encoder.putString(b.name, "NAME")
                encoder.putString(b.parent.name if b.parent else "", "PARENT")
                encoder.putNumber(b.length, "LENGTH")

                encoder.beginBlock()

                if (b.parent):
                    mat = convMat @ (obj.matrix_world @ (b.parent.matrix.inverted() @ b.matrix) @ obj.matrix_world.inverted()) @ convMatInv
                else:
                    mat = convMat @ (obj.matrix_world @ b.matrix @ obj.matrix_world.inverted()) @ convMatInv
                
                encoder.printf("%f, %f, %f, %f,\t%f, %f, %f, %f,\t%f, %f, %f, %f;", *mat[0], *mat[1], *mat[2])

                encoder.endBlock()

            bpy.ops.object.mode_set(mode="OBJECT")

            encoder.putValue("SE_SKELETON_END")

def saveAnimsetBin(filename: str, armObj: object, convMat: Matrix, convMatInv: Matrix, ver: str):
    anims = toSkaAnimset(armObj, convMat, convMatInv)

    version = {
        "1.07": 14,
        ">1.07": 14,
        "LC": 14,
    }[ver]

    ska.writeAnimsetBin(anims, filename, version)

def saveAnimsetAscii(filename: str, armObj: object, convMat: Matrix, convMatInv: Matrix):
    if (armObj):
        bpy.context.view_layer.objects.active = armObj
        bpy.ops.object.mode_set(mode="POSE")

        bpy.ops.pose.transforms_clear()

    for action in bpy.data.actions:
        name = action.name
        shapeAction = False
        if (action.name.startswith("SK_")):
            armAction = bpy.data.actions.get(action.name[3:], None)
            if (not armAction):
                name = action.name[3:]
                shapeAction = True
            else:
                continue

        secPerFrame = 1/30
        frameCount = int(action.frame_range[1] - action.frame_range[0]) + 1

        if (shapeAction):
            skaction = action
        else:
            skaction = bpy.data.actions.get("SK_{}".format(action.name), None)

        shapeKeys = dict()

        if (skaction):
            for fc in skaction.fcurves:
                if (fc.data_path.startswith("key_blocks")):
                    mapName = fc.data_path[12:fc.data_path.find('"', 12)]
                    factors = [None for i in range(frameCount)]

                    shapeKeys[mapName] = factors
                    for kp in fc.keyframe_points:
                        factors[int(kp.co[0]) - int(action.frame_range[0])] = kp.co[1]

                    prev = None
                    for i in range(frameCount):
                        f = factors[i]
                        if (f is None):
                            nextFactor = None
                            idx = -1
                            for j, f2 in enumerate(factors[i+1:]):
                                if (not f2 is None):
                                    nextFactor = f2
                                    idx = i + j + 1
                                    break
                            if (nextFactor is None and prev is None):
                                break
                            elif (not nextFactor is None and prev is None):
                                for k in range(idx):
                                    factors[k] = nextFactor
                            elif (nextFactor is None and not prev is None):
                                for k in range(i, frameCount):
                                    factors[k] = prev
                            else:
                                step = 1 / (idx-i+1)
                                lerp = step
                                for k in range(i, idx):
                                    factors[k] = prev + (nextFactor - prev) * lerp
                                    lerp += step
                        prev = f

        if (armObj or shapeKeys):
            name = action.name.replace("|", "").replace("/", "").replace("\\", "")
            fn = "{}_{}.aa".format(filename, name)

            if (not armObj.animation_data):
                armObj.animation_data_create()

            with open(fn, "w") as file:
                encoder = asciiparser.AsciiEncoder(file)
                encoder.putNumber(0.1, "SE_ANIM")

                encoder.putNumber(secPerFrame, "SEC_PER_FRAME")
                encoder.putNumber(frameCount, "FRAMES")
                encoder.putString(action.name, "ANIM_ID")

                if (armObj and armObj.pose.bones):
                    armObj.animation_data.action = action
                    for bone in encoder.iterList(armObj.pose.bones, "BONEENVELOPES"):
                        if (bone.parent):
                            relMat = convMat @ (armObj.matrix_world @ (bone.parent.bone.matrix_local.inverted() @ bone.bone.matrix_local) @ armObj.matrix_world.inverted()) @ convMatInv
                        else:
                            relMat = convMat @ (armObj.matrix_world @ bone.bone.matrix_local @ armObj.matrix_world.inverted()) @ convMatInv

                        encoder.putString(bone.name, "NAME", "")

                        for i in encoder.iterList([0], "DEFAULT_POSE", False):
                            encoder.printf("%f, %f, %f, %f,\t%f, %f, %f, %f,\t%f, %f, %f, %f;", *relMat[0], *relMat[1], *relMat[2])

                        encoder.beginBlock()
                        for i in range(int(action.frame_range[0]), int(action.frame_range[1])+1):
                            bpy.context.scene.frame_set(i)
                            if (bone.parent):
                                mat = convMat @ (armObj.matrix_world @ (bone.parent.matrix.inverted() @ bone.matrix) @ armObj.matrix_world.inverted()) @ convMatInv
                            else:
                                mat = convMat @ (armObj.matrix_world @ bone.matrix @ armObj.matrix_world.inverted()) @ convMatInv
                            encoder.printf("%f, %f, %f, %f,\t%f, %f, %f, %f,\t%f, %f, %f, %f;", *mat[0], *mat[1], *mat[2])
                        encoder.endBlock()
                        bpy.context.scene.frame_set(0)
                else:
                    for i in encoder.iterList([], "BONEENVELOPES"):
                        pass

                for name in encoder.iterList(shapeKeys, "MORPHENVELOPES"):
                    encoder.putString(name, "NAME", "")

                    fs = shapeKeys[name]

                    encoder.beginBlock()
                    for f in fs:
                        encoder.printf("%f;", f)
                    encoder.endBlock()

                encoder.putValue("SE_ANIM_END")

    if (bpy.data.objects):
        bpy.ops.object.mode_set(mode="OBJECT")

def save(filename: str, binary: bool, makeSmc: bool, saveMesh: bool, saveSkeleton: bool, saveAnimset: bool, meshtype: int, selected: bool, forward: str, up: str, ver: str, handleMeshes: str, handleArmatures: str):
    convMat = axis_conversion("Y", "Z", forward, up).to_4x4()
    convMatInv = convMat.inverted()

    meshObjects = []
    armObjects = []
    for obj in bpy.context.scene.objects:
        if (not selected or obj.select_get()):
            if (obj.type == "MESH"):
                meshObjects.append(obj)
            elif (obj.type == "ARMATURE"):
                armObjects.append(obj)

    meshFns = []
    skeletonFns = []
    meshAsLODs = handleMeshes == "LOD"
    armAsLODs = handleArmatures == "LOD"
    animsetFn = ""
    if (meshObjects and saveMesh):
        if (binary):
            if (meshAsLODs):
                meshFn = filename + ".bm"
                saveMeshBin(meshFn, meshObjects, meshtype, convMat, ver)
                meshFns.append(meshFn)
            else:
                for obj in meshObjects:
                    meshFn = f"{filename}_{obj.name}.bm"
                    saveMeshBin(meshFn, [obj], meshtype, convMat, ver)
                    meshFns.append(meshFn)
        else:
            saveMeshAscii(filename, meshObjects, meshtype, convMat)

    if (armObjects and saveSkeleton):
        if (binary):
            if (armAsLODs):
                skeletonFn = filename + ".bs"
                saveSkeletonBin(skeletonFn, armObjects, convMat, convMatInv, ver)
                skeletonFns.append(skeletonFn)
            else:
                for obj in armObjects:
                    skeletonFn = f"{filename}_{obj.name}.bs"
                    saveSkeletonBin(skeletonFn, [obj], convMat, convMatInv, ver)
                    skeletonFns.append(skeletonFn)
        else:
            saveSkeletonAscii(filename, armObjects, convMat, convMatInv)

    if (saveAnimset):
        armObj = None
        if (armObjects):
            armObj = max(armObjects, key=lambda obj: len(obj.data.bones))

        if (binary):
            animsetFn = filename + ".ba"
            saveAnimsetBin(animsetFn, armObj, convMat, convMatInv, ver)
        else:
            saveAnimsetAscii(filename, armObj, convMat, convMatInv)

    if (makeSmc and binary):
        smcFn = filename + ".smc"
        root = utils.getGameRoot(filename)
        if (not root):
            raise RuntimeError("Could not find game root from '{}', you sure this file located inside Serious Engine game folder?".format(filename))

        modelName = os.path.basename(filename)
        with open(smcFn, "w") as file:
            file.write('NAME "{}";\n{{\n'.format(modelName))
            for meshFn in meshFns:
                file.write('\tMESH\tTFNM "{}";\n'.format(meshFn.replace(root, "")[1:]))
            for skeletonFn in skeletonFns:
                file.write('\tSKELETON\tTFNM "{}";\n'.format(skeletonFn.replace(root, "")[1:]))
            if (animsetFn):
                file.write('\tANIMSET\tTFNM "{}";\n'.format(animsetFn.replace(root, "")[1:]))
            file.write("}")